#!/bin/bash

CONFIG_FILE="config.ini"

# Load config values
get_config_value() {
    section=$1
    key=$2
    awk -F '=' -v section="$section" -v key="$key" '
    $0 ~ "\\[" section "\\]" { in_section=1; next }
    in_section && $1 ~ key { gsub(/ /, "", $2); print $2; exit }
    ' "$CONFIG_FILE"
}

# Read config values
OPENAM_AUTHENTICATE_URL=$(get_config_value "OpenAM" "openam_authenticate_url")
OPENAM_USERNAME=$(get_config_value "OpenAM" "username")
OPENAM_PASSWORD=$(get_config_value "OpenAM" "password")

OIDC_SERVER_URL=$(get_config_value "OAuth" "server_url")
OIDC_REALM=$(get_config_value "OAuth" "realm")
OIDC_CLIENT_ID=$(get_config_value "OAuth" "client_id")
OIDC_CLIENT_SECRET=$(get_config_value "OAuth" "client_secret")
OIDC_SCOPES=$(get_config_value "OAuth" "scopes")

PORTAL_URL=$(get_config_value "Settings" "portal_url")
PRODUCT_URL_OLD=$(get_config_value "Settings" "product_url_old")
PRODUCT_URL_NEW=$(get_config_value "Settings" "product_url_new")
SSO_COOKIENAME=$(get_config_value "Settings" "sso_cookiename")
PRINT_ERRORS=$(get_config_value "Settings" "print_errors")

# Show success banner
show_success() {
    echo
    printf '%*s\n' 80 '' | tr ' ' '='
    echo "[SUCCESS] Source: $1"
    echo "URL: $2"
    printf '%*s\n' 80 '' | tr ' ' '-'
    echo "$3"
    printf '%*s\n' 80 '' | tr ' ' '='
    echo
}

# Get OpenAM auth cookie
get_auth_cookie() {
    response=$(curl -s -X POST "$OPENAM_AUTHENTICATE_URL" \
        -H "Content-Type: application/json" \
        -H "X-OpenAM-Username: $OPENAM_USERNAME" \
        -H "X-OpenAM-Password: $OPENAM_PASSWORD" \
        -d '{}')

    token=$(echo "$response" | jq -r '.tokenId')
    if [[ "$token" != "null" ]]; then
        echo "$token"
    else
        [[ "$PRINT_ERRORS" == "true" ]] && echo "OpenAM authentication failed" >&2
        echo ""
    fi
}

# Get OAuth2 token
get_oidc_token() {
    token_url="$OIDC_SERVER_URL/realms/$OIDC_REALM/protocol/openid-connect/token"
    data="grant_type=client_credentials&client_id=$OIDC_CLIENT_ID&client_secret=$OIDC_CLIENT_SECRET"
    [[ -n "$OIDC_SCOPES" && "$OIDC_SCOPES" != "openid" ]] && data+="&scope=$OIDC_SCOPES"

    response=$(curl -s -X POST "$token_url" -d "$data")
    access_token=$(echo "$response" | jq -r '.access_token')
    echo "$access_token"
}

# Check if URL is OIDC protected
is_oidc_protected() {
    status=$(curl -s -o /dev/null -w "%{http_code}" -H "Authorization: Bearer TEST" "$1")
    [[ "$status" == "401" ]]
}

# Get product data via OIDC
get_product_data_oidc() {
    curl -s -H "Authorization: Bearer $1" "$2"
}

# Get product data via OpenAM
get_product_data_am() {
    curl -s --cookie "$SSO_COOKIENAME=$1" "$2"
}

# Main logic
echo "Attempting OpenID authentication"
access_token=$(get_oidc_token)

for url in "$PRODUCT_URL_NEW" "$PRODUCT_URL_OLD"; do
    if is_oidc_protected "$url"; then
        echo "$url appears to be OIDC-protected (401 observed)."
        if [[ -z "$access_token" ]]; then
            echo "OIDC detected but no usable access token; exiting."
            exit 1
        fi
        content=$(get_product_data_oidc "$access_token" "$url")
        show_success "OIDC" "$url" "$content"
        exit 0
    else
        echo "$url does not appear to be OIDC-protected (401 not observed)."
    fi
done

echo "Attempting legacy OpenAM authentication"
auth_cookie=$(get_auth_cookie)
if [[ -n "$auth_cookie" ]]; then
    echo "OpenAM authentication successful. Obtained authentication cookie: $SSO_COOKIENAME=$auth_cookie"
    content=$(get_product_data_am "$auth_cookie" "$PRODUCT_URL_NEW")
    show_success "OpenAM" "$PRODUCT_URL_NEW" "$content"
    [[ "$PRODUCT_URL_OLD" != "$PRODUCT_URL_NEW" ]] && {
        content=$(get_product_data_am "$auth_cookie" "$PRODUCT_URL_OLD")
        show_success "OpenAM" "$PRODUCT_URL_OLD" "$content"
    }
else
    echo "OpenAM authentication failed."
fi