#!/usr/bin/python
# -*- coding: utf-8 -*-

import configparser
import json
import sys
from datetime import datetime, timezone
from http import HTTPStatus

import requests


def load_config():
    config = configparser.ConfigParser()
    config.read('config.ini')
    return config


config = load_config()
print(config.sections())

# --- OpenAM configuration ---
OPENAM_AUTHENTICATE_URL = config['OpenAM']['openam_authenticate_url']
openam_username = config['OpenAM']['username']
openam_password = config['OpenAM']['password']

# --- OAuth/OIDC configuration ---
OIDC_SERVER_URL = config['OAuth']['server_url']
OIDC_REALM = config['OAuth']['realm']
OIDC_CLIENT_ID = config['OAuth']['client_id']
OIDC_CLIENT_SECRET = config['OAuth']['client_secret']
OIDC_SCOPES = config['OAuth']['scopes']

# --- General settings ---
PORTAL_URL = config['Settings']['portal_url']
PRODUCT_URL_OLD = config['Settings']['product_url_old']
PRODUCT_URL_NEW = config['Settings']['product_url_new']
SSO_COOKIENAME = config['Settings']['sso_cookiename']
PRINT_ERRORS = config.getboolean('Settings', 'print_errors')  # controls stdout error output


def show_success(source: str, url: str, content: str):
    """
    Print a clear banner and the retrieved content to stdout (no exit here).
    """
    print("\n" + "=" * 80)
    print(f"[SUCCESS] Source: {source} | URL: {url}")
    print("-" * 80)
    print(content)
    print("=" * 80 + "\n")


def get_auth_cookie(username, password):
    """
    Authenticates a user against OpenAM. This uses the globally defined AUTHENTICATE_URL.
    It returns whether authentication was successful and if so, the obtained auth cookie.
    If an error occurs, the exception is caught and the error printed to stdout.

    :param str username: The username to use for authentication.
    :param str password: The password for the provided username.

    :returns (bool, str): (True, cookie) if successful; otherwise (False, '').
    """
    try:
        response = requests.post(
            OPENAM_AUTHENTICATE_URL,
            headers={
                'Content-Type': 'application/json',
                'X-OpenAM-Username': username,
                'X-OpenAM-Password': password,
            },
            data='{}',
        )
        token_dict = response.json()
        auth_cookie = token_dict['tokenId']
        return True, auth_cookie
    except Exception as exc:
        if PRINT_ERRORS:
            print(exc)
        return False, ''


def get_oauth2_access_token(client_id: str, client_secret: str, scopes: str):
    try:
        token_endpoint_url = f"{OIDC_SERVER_URL}/realms/{OIDC_REALM}/protocol/openid-connect/token"

        data = {
            "grant_type": "client_credentials",
            "client_id": client_id,
            "client_secret": client_secret,
        }
        # Only send scope if it's truthy and not the user-login scope
        if scopes and scopes.strip() and scopes.strip() != "openid":
            data["scope"] = scopes  # space-separated client scope names

        response = requests.post(token_endpoint_url, data=data, timeout=15)

        if response.status_code != HTTPStatus.OK:
            if PRINT_ERRORS:
                print(response.status_code, response.text)  # see exact IdP error
            return None, None

        return response.json(), datetime.now(timezone.utc)
    except Exception as e:
        if PRINT_ERRORS:
            print("Requesting access token has failed: " + str(e))
        return None, None


def is_oidc_protected(url, token="TEST", timeout=10):
    """
    Send a GET with a Bearer token. If the server replies 401, assume OIDC.
    Returns True for OIDC, False for OpenAM/other.
    """
    try:
        r = requests.get(
            url,
            headers={"Authorization": f"Bearer {token}"},
            timeout=timeout,
            allow_redirects=True,
        )
        return r.status_code == HTTPStatus.UNAUTHORIZED
    except Exception:
        # On network/SSL errors, default to OpenAM/other
        return False


def get_product_data_oidc(access_token, product_url):
    try:
        test_response = requests.get(
            product_url,
            headers={"Authorization": f"Bearer {access_token}"},
        )
        return True, test_response.text
    except Exception as exc:
        if PRINT_ERRORS:
            print(exc)
    return False, {}


def get_product_data_am(auth_token, product_url):
    try:
        test_response = requests.get(
            product_url,
            cookies={SSO_COOKIENAME: auth_token},
        )
        return True, test_response.text
    except Exception as exc:
        if PRINT_ERRORS:
            print(exc)
    return False, {}


def retrieve_data_behind_sso():
    # Track which URLs succeeded and by which method
    url_success = {PRODUCT_URL_OLD: False, PRODUCT_URL_NEW: False}
    url_succeeded_via_oidc = {PRODUCT_URL_OLD: False, PRODUCT_URL_NEW: False}

    # --- OIDC/OAuth2 authentication (preferred) ---
    access_token = None
    if OIDC_CLIENT_ID and OIDC_CLIENT_SECRET:
        print("Attempting OpenID authentication")
        token_incl_metadata, response_timestamp = get_oauth2_access_token(
            OIDC_CLIENT_ID, OIDC_CLIENT_SECRET, OIDC_SCOPES
        )
        if token_incl_metadata is not None:
            age_seconds = (datetime.now(timezone.utc) - response_timestamp).total_seconds()
            has_token_expired = age_seconds > token_incl_metadata.get("expires_in")
            if has_token_expired:
                print("Token has expired and needs to be renewed")
            else:
                print('OpenID/OAuth2 authentication successful')
                print('Received token: ')
                print(json.dumps(token_incl_metadata, indent=4))
                access_token = token_incl_metadata.get("access_token")

    # ---- OIDC check: NEW first, then OLD; EXIT if any URL is OIDC ----
    for url in [PRODUCT_URL_NEW, PRODUCT_URL_OLD]:
        if is_oidc_protected(url):
            print(f"{url} appears to be OIDC-protected (401 observed).")
            if not access_token:
                print("OIDC detected but no usable access token; exiting.")
                sys.exit(1)
            ok, content = get_product_data_oidc(access_token, url)
            print(f"OIDC check for {url}: {ok}")
            if ok:
                show_success("OIDC", url, content)
                url_success[url] = True
                url_succeeded_via_oidc[url] = True
                sys.exit(0)
            else:
                print(f"OIDC detected on {url} but content retrieval failed. Exiting.")
                sys.exit(1)
        else:
            print(f"{url} does not appear to be OIDC-protected (401 not observed).")

    # --- OpenAM fallback ONLY if both URLs are not OIDC ---
    if openam_username and openam_password:
        print("Attempting legacy OpenAM authentication")
        authenticated, auth_cookie = get_auth_cookie(openam_username, openam_password)
        if authenticated:
            print('OpenAM authentication successful. Obtained authentication cookies:')
            print(f'  {SSO_COOKIENAME}: {auth_cookie}')

            # Try NEW first
            ok, content = get_product_data_am(auth_cookie, PRODUCT_URL_NEW)
            print(f"OpenAM check for {PRODUCT_URL_NEW}: {ok}")
            if ok:
                show_success("OpenAM", PRODUCT_URL_NEW, content)
                url_success[PRODUCT_URL_NEW] = True

            # Try OLD only if different
            if PRODUCT_URL_OLD != PRODUCT_URL_NEW:
                ok, content = get_product_data_am(auth_cookie, PRODUCT_URL_OLD)
                print(f"OpenAM check for {PRODUCT_URL_OLD}: {ok}")
                if ok:
                    show_success("OpenAM", PRODUCT_URL_OLD, content)
                    url_success[PRODUCT_URL_OLD] = True
        else:
            print("OpenAM authentication failed.")


def main():
    retrieve_data_behind_sso()


if __name__ == "__main__":
    main()
